// Репозиторий настроек

function Settings() {
    /**
     * сами настройки
     * @type {{}}
     * @private
     */
    let _settings = {
        foo: 'bar',
    };

    /**
     * подписчики на события
     * @type {{}}
     * @private
     */
    let _subscribers = {};

    /**
     * загружены ли настройки из хранилища
     * @type {boolean}
     * @private
     */
    let _ready = false;
    _init();

    /**
     * Инициализация класса
     * @private
     */
    function _init() {
        chrome.storage.local.onChanged.addListener(function (changes, namespace) {
            if (changes.settings === undefined || changes.settings.newValue === undefined) {
                return;
            }
            _settings = changes.settings.newValue;
            _emit({name: 'update', data: _settings});
        });
        _load();
    }

    /**
     * загрузка настроек из хранилища
     * @private
     */
    function _load() {
        chrome.storage.local.get(['settings'], function (result) {
            if (result.settings) {
                _settings = result.settings;
            }
            _ready = true;
            _emit({name: 'ready', data: _settings});
        });
    }

    function save() {
        chrome.storage.local.set({settings: _settings});
    }

    function get(key) {
        return _settings[key];
    }

    function set(key, value) {
        _settings[key] = value;
    }

    function ready() {
        return _ready;
    }

    function setMany(data) {
        _settings = {..._settings, ...data};
    }

    function getAll() {
        return _settings;
    }

    /**
     * подписаться на событие настроек (загружены, обновлены)
     * @param eventName 'ready'|'update'
     * @param listener
     */
    function on(eventName, listener) {
        if (_subscribers[eventName] === undefined) {
            _subscribers[eventName] = [];
        }
        _subscribers[eventName].push(listener);
    }

    /**
     * отписаться от события настроек
     * @param eventName
     * @param listener
     */
    function off(eventName, listener) {
        if (_subscribers[eventName] === undefined) {
            return;
        }
        let keyToDelete = null;
        _subscribers[eventName].forEach((element, index) => {
            if (element === listener) {
                keyToDelete = index;
            }
        });
        if (keyToDelete !== null) {
            _subscribers[eventName].splice(keyToDelete, 1);
        }
    }

    function _emit(event) {
        const eventName = event.name;
        if (_subscribers[eventName]) {
            for (let listener of _subscribers[eventName]) {
                listener(event);
            }
        }
    }

    return {
        get,
        set,
        ready,
        save,
        setMany,
        getAll,
        on,
        off,
    };
}

const settings = new Settings();
